"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _authentication_error = _interopRequireDefault(require("../auth/errors/authentication_error"));
var _user = _interopRequireDefault(require("../auth/user"));
var _multitenancy = require("../../../../common/multitenancy");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The SearchGuard  backend.
 */
class SearchGuardBackend {
  constructor({
    elasticsearch,
    configService,
    core
  }) {
    (0, _defineProperty2.default)(this, "_client", async ({
      headers = {},
      asWho = 'asCurrentUser',
      ...options
    }) => {
      const result = await this.elasticsearch.client.asScoped({
        headers
      })[asWho].transport.request(options);
      return result;
    });
    (0, _defineProperty2.default)(this, "getAuthConfig", async (nextUrl = null) => {
      try {
        const sgFrontendConfigId = this.configService.get('searchguard.sg_frontend_config_id') || 'default';
        let frontendBaseUrl = this.configService.get('searchguard.frontend_base_url') || this.core.http.basePath.publicBaseUrl;
        if (!frontendBaseUrl) {
          let serverInfo = this.core.http.getServerInfo();
          frontendBaseUrl = serverInfo.protocol + "://" + serverInfo.hostname + ":" + serverInfo.port + "/" + this.core.http.basePath.serverBasePath;
        }
        const response = await this._client({
          path: '/_searchguard/auth/config',
          method: 'POST',
          asWho: 'asInternalUser',
          body: {
            config_id: sgFrontendConfigId,
            frontend_base_url: frontendBaseUrl,
            next_url: nextUrl
          }
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "authenticate", async credentials => {
      const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');
      try {
        const response = await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers: {
            authorization: `Basic ${authHeader}`
          }
        });
        return new _user.default(credentials.username, credentials, credentials, response.sg_roles, response.backend_roles, response.sg_tenants, response.user_requested_tenant);
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "authenticateWithHeader", async (headerName, headerValue, additionalAuthHeaders = {}) => {
      try {
        const credentials = {
          headerName: headerName,
          headerValue: headerValue
        };
        const headers = {
          ...additionalAuthHeaders
        };

        // For anonymous auth, we wouldn't have any value here
        if (headerValue) {
          headers[headerName] = headerValue;
        }
        const response = await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers
        });
        return new _user.default(response.user_name, credentials, null, response.sg_roles, response.backend_roles, response.sg_tenants, response.user_requested_tenant);
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    /**
     * A wrapper for authinfo() when we expect a response to be used for a cookie
     * @param headers
     * @param credentials
     * @returns {Promise<User>}
     */
    (0, _defineProperty2.default)(this, "authenticateWithHeaders", async (headers, credentials = {}, additionalAuthHeaders = {}) => {
      headers = {
        ...additionalAuthHeaders,
        ...headers
      };
      try {
        const response = await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers
        });
        return new _user.default(response.user_name, credentials, null, response.sg_roles, response.backend_roles, response.sg_tenants, response.user_requested_tenant);
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "createSessionWithHeaders", async (headers, additionalAuthHeaders = {}) => {
      headers = {
        ...additionalAuthHeaders,
        ...headers
      };
      try {
        return await this._client({
          path: '/_searchguard/auth/session/with_header',
          method: 'POST',
          headers
        });
      } catch (error) {
        console.log(error);
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "authinfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "sessionInfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/auth/session',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * NB: type not complete
     * @typedef KibanaInfo
     * @prop {string} user_name
     * @prop {boolean} not_fail_on_forbidden_enabled
     * @prop {boolean} kibana_mt_enabled
     */
    /**
     *
     * @returns {Promise<KibanaInfo>}
     */
    (0, _defineProperty2.default)(this, "getKibanaInfoWithInternalUser", async () => {
      try {
        return await this._client({
          path: '/_searchguard/kibanainfo',
          method: 'get',
          asWho: 'asInternalUser'
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * Check for application permissions
     * @param headers
     * @param permissions
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "hasPermissions", async (headers, permissions) => {
      try {
        return await this._client({
          path: '/_searchguard/permission',
          method: 'get',
          headers,
          querystring: {
            permissions
          }
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "multitenancyinfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/kibanainfo',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "systeminfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/license',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getTenantInfoWithInternalUser", async () => {
      try {
        return await this._client({
          path: '/_searchguard/tenantinfo',
          method: 'get',
          asWho: 'asInternalUser'
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getTenantInfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/tenantinfo',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * @typedef UserTenant
     * @prop {boolean} read_access
     * @prop {boolean} write_access
     * @prop {boolean} exists
     */
    /**
     * @typedef UserTenantInfo
     * @prop {number} status
     * @prop {object} data
     * @prop {boolean} data.multi_tenancy_enabled
     * @prop {string} data.username
     * @prop {string?} data.default_tenant
     * @prop {string|null} data.user_requested_tenant
     * @prop {Record<string, UserTenant>} data.tenants
     */
    /**
     *
     * @param headers
     * @returns {Promise<UserTenantInfo>}
     */
    (0, _defineProperty2.default)(this, "getUserTenantInfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/current_user/tenants',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * The user tenant info endpoint contains information about
     * read/write access, as well as an exists flag which
     * is false if the tenant is empty.
     *
     * This function filters out tenants that
     * are read only and does not exist.
     *
     * This is to prevent that a user ends up in
     * an empty tenant with only read access
     *
     * @param {UserTenantInfo} userTenantInfo
     * @returns {UserTenantInfo}
     */
    (0, _defineProperty2.default)(this, "removeNonExistingReadOnlyTenants", userTenantInfo => {
      if (userTenantInfo.data && userTenantInfo.data.tenants) {
        Object.keys(userTenantInfo.data.tenants).forEach(key => {
          if (userTenantInfo.data.tenants[key].write_access !== true && userTenantInfo.data.tenants[key].exists !== true) {
            delete userTenantInfo.data.tenants[key];
          }
        });
      }
      return userTenantInfo;
    });
    /**
     * Converts the UserTenantInfo tenants to the tenantName = write_access format
     * to stay compatible with existing code
     * @param {Record<string, UserTenant>} userTenants
     * @return {Record<string, boolean>}
     */
    (0, _defineProperty2.default)(this, "convertUserTenantsToRecord", userTenants => {
      /**
       * @type {Record<string, boolean>}
       */
      const tenantsRecord = {};
      Object.keys(userTenants).forEach(tenant => {
        tenantsRecord[tenant] = userTenants[tenant].write_access;
      });
      return tenantsRecord;
    });
    (0, _defineProperty2.default)(this, "uploadLicense", async (headers, body) => {
      try {
        return await this._client({
          path: '/_searchguard/api/license',
          method: 'put',
          headers,
          body
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * @deprecated, use the sessionPlugin instead
     * @param user
     * @returns {Promise<{authorization: string}>}
     */
    (0, _defineProperty2.default)(this, "getAuthHeaders", async user => {
      const credentials = user.credentials;
      const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');
      return {
        authorization: `Basic ${authHeader}`
      };
    });
    this.elasticsearch = elasticsearch;
    this.configService = configService;
    this.core = core;
  }
  async authenticateWithSession(credentials) {
    try {
      const response = await this._client({
        path: '/_searchguard/auth/session',
        method: 'POST',
        body: credentials
      });
      return response;
    } catch (error) {
      // TODO remove
      console.log(error);
      throw error;
    }
  }
  async logoutSession(headers) {
    try {
      return await this._client({
        path: '/_searchguard/auth/session',
        method: 'DELETE',
        headers
      });
    } catch (error) {
      if (error.statusCode === 401) {
        throw new _authentication_error.default('Invalid username or password', error);
      }
      throw error;
    }
  }
  buildSessionResponse(credentials, authInfoResponse) {
    return new _user.default(authInfoResponse.user_name, credentials, null, authInfoResponse.sg_roles, authInfoResponse.backend_roles, authInfoResponse.sg_tenants, authInfoResponse.user_requested_tenant);
  }
  getAuthHeaders(username, password) {
    const authHeader = Buffer.from(`${username}:${password}`).toString('base64');
    return {
      authorization: `Basic ${authHeader}`
    };
  }
  getUser(username, password) {
    const credentials = {
      username: username,
      password: password
    };
    const user = new _user.default(credentials.username, credentials, credentials, [], {});
    return user;
  }
  updateAndGetTenantPreferences(request, user, tenant) {
    /*
    const preferencesCookieName = this.configService.get(
      'searchguard.cookie.preferences_cookie_name'
    );
      */

    //const prefs = request.state[preferencesCookieName];
    const prefs = {};
    // no prefs cookie present
    if (!prefs) {
      const newPrefs = {};
      newPrefs[user] = tenant;
      return newPrefs;
    }
    prefs[user] = tenant;
    return prefs;
  }
  getTenantByPreference(request, username, tenants, preferredTenants, globalEnabled, privateEnabled) {
    // delete user from tenants first to check if we have a tenant to choose from at all
    // keep original preferences untouched, we need the original values again
    // http://stackoverflow.com/questions/728360/how-do-i-correctly-clone-a-javascript-object
    const tenantsCopy = JSON.parse(JSON.stringify(tenants));
    delete tenantsCopy[username];

    // We have two paths for deciding if the global tenant is available:
    // searchguard.multitenancy.global.enabled and authinfo.sg_tenants
    if (!tenantsCopy.hasOwnProperty(_multitenancy.GLOBAL_TENANT_NAME)) {
      globalEnabled = false;
    }

    // Make sure we sync with the backend
    if (!globalEnabled) {
      delete tenantsCopy[_multitenancy.GLOBAL_TENANT_NAME];
    }

    // sanity check
    if (!globalEnabled && !privateEnabled && _lodash.default.isEmpty(tenantsCopy)) {
      return null;
    }

    // Evaluate preferredTenants from kibana config
    if (preferredTenants && !_lodash.default.isEmpty(preferredTenants)) {
      for (let i = 0; i < preferredTenants.length; i++) {
        //const check = preferredTenants[i].toLowerCase();
        const check = preferredTenants[i];
        if (globalEnabled && (check.toLowerCase() === 'global' || check.toLowerCase() === '__global__')) {
          return _multitenancy.GLOBAL_TENANT_NAME;
        }
        if (privateEnabled && (check.toLowerCase() === 'private' || check.toLowerCase() === '__user__') && tenants[username] !== undefined) {
          return '__user__';
        }

        // TODO Test if SGS_GLOBAL_TENANT is handled correctly
        if (tenantsCopy[check] !== undefined) {
          return check;
        }
        if (tenantsCopy[check.toLowerCase()] !== undefined) {
          return check.toLowerCase();
        }
        if (check.toLowerCase() === 'private' && privateEnabled) {
          return '__user__';
        }
      }
    }

    // no pref in cookie, no preferred tenant in kibana, use GLOBAL, Private or the first tenant in the list
    if (globalEnabled) {
      return _multitenancy.GLOBAL_TENANT_NAME;
    }
    if (privateEnabled) {
      return '__user__';
    } else {
      delete tenants[username];
    }

    // sort tenants by putting the keys in an array first
    let tenantkeys = [];
    let k;
    for (k in tenants) {
      if (tenants.hasOwnProperty(k)) {
        tenantkeys.push(k);
      }
    }
    tenantkeys.sort();
    if (!globalEnabled) {
      tenantkeys = tenantkeys.filter(tenantKey => tenantKey !== _multitenancy.GLOBAL_TENANT_NAME);
    }
    if (tenantkeys.length) {
      return tenantkeys[0];
    }
    return null;
  }

  /**
   *
   * @param username
   * @param requestedTenant
   * @param {Record<string, boolean>} tenants
   * @returns {string|null}
   */
  validateRequestedTenant(username, requestedTenant, tenants) {
    // Translate the private tenant
    if ((requestedTenant === _multitenancy.PRIVATE_TENANT_NAME || requestedTenant === 'private') && typeof tenants[username] !== 'undefined') {
      return _multitenancy.PRIVATE_TENANT_NAME;
    }

    // This should not really happen, but if for some reason
    // the username ends up in the cookie, we translate
    if (requestedTenant === username) {
      return _multitenancy.PRIVATE_TENANT_NAME;
    }
    if (tenants && typeof tenants[requestedTenant] !== 'undefined') {
      return requestedTenant;
    }
    return null;
  }

  /**
   *
   * @param username
   * @param requestedTenant
   * @param {Record<string, boolean>} tenants
   * @param globalEnabled
   * @param privateEnabled
   * @returns {*|string|null}
   */
  validateTenant(username, requestedTenant, tenants, globalEnabled, privateEnabled) {
    // delete user from tenants first to check if we have a tenant to choose from at all
    // keep original preferences untouched, we need the original values again
    // http://stackoverflow.com/questions/728360/how-do-i-correctly-clone-a-javascript-object
    const tenantsCopy = JSON.parse(JSON.stringify(tenants));
    delete tenantsCopy[username];
    console.log('>>>>> Validating tenants', requestedTenant, tenants);

    // We have two paths for deciding if the global tenant is available:
    // searchguard.multitenancy.global.enabled and authinfo.sg_tenants
    if (!tenantsCopy.hasOwnProperty(_multitenancy.GLOBAL_TENANT_NAME)) {
      globalEnabled = false;
    }
    if (!globalEnabled) {
      delete tenantsCopy[_multitenancy.GLOBAL_TENANT_NAME];
    }

    // sanity check: no global, no private, no other tenants -> no tenant available
    if (!globalEnabled && !privateEnabled && _lodash.default.isEmpty(tenantsCopy)) {
      return null;
    }

    // requested tenant accessible for user
    // TODO do we need to check lowercase here...? Not really, tenants are case sensitive
    if (tenantsCopy[requestedTenant] !== undefined) {
      return requestedTenant;
    }

    // Using tenants instead of tenantsCopy here is intentional
    // because the private tenant is always deleted from the tenantsCopy
    if ((requestedTenant === _multitenancy.PRIVATE_TENANT_NAME || requestedTenant === 'private') && tenants[username] && privateEnabled) {
      return _multitenancy.PRIVATE_TENANT_NAME;
    }

    // This is the path when we have a tenant named global in the query parameter
    if ((requestedTenant === 'global' || requestedTenant === '') && globalEnabled) {
      return _multitenancy.GLOBAL_TENANT_NAME;
    }
    return null;
  }
}
exports.default = SearchGuardBackend;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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